<?php
require_once __DIR__ . '/../includes/config.php';

$message = '';
$messageType = '';
$uploadDir = __DIR__ . '/';
$logosDir = __DIR__ . '/logos/';

// Ensure directories exist
if (!is_dir($logosDir)) {
    mkdir($logosDir, 0775, true);
}

// Handle file upload
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['file'])) {
    $targetDir = isset($_POST['upload_type']) && $_POST['upload_type'] === 'logo' ? $logosDir : $uploadDir;
    
    $file = $_FILES['file'];
    $fileName = basename($file['name']);
    $targetFile = $targetDir . $fileName;
    
    // Validate file
    $allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp', 'application/pdf'];
    $maxSize = 10 * 1024 * 1024; // 10MB
    
    if ($file['size'] > $maxSize) {
        $message = 'File too large. Maximum size is 10MB.';
        $messageType = 'error';
    } elseif (!in_array($file['type'], $allowedTypes)) {
        $message = 'Invalid file type. Allowed: JPG, PNG, GIF, WebP, PDF';
        $messageType = 'error';
    } else {
        // Sanitize filename
        $fileName = preg_replace('/[^a-zA-Z0-9._-]/', '', $fileName);
        $targetFile = $targetDir . $fileName;
        
        if (move_uploaded_file($file['tmp_name'], $targetFile)) {
            $message = 'File uploaded successfully: ' . $fileName;
            $messageType = 'success';
        } else {
            $message = 'Upload failed. Please try again.';
            $messageType = 'error';
        }
    }
}

// Get existing files
$logos = glob($logosDir . '*.{jpg,jpeg,png,gif,webp}', GLOB_BRACE);
$files = glob($uploadDir . '*.{jpg,jpeg,png,gif,webp,pdf}', GLOB_BRACE);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>File Upload - Multicure Network</title>
    <link rel="stylesheet" href="../css/style.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    <style>
        .upload-page { min-height: 100vh; background: var(--gray-50); padding: 40px 0; }
        .upload-container { max-width: 900px; margin: 0 auto; padding: 0 24px; }
        .upload-card { background: white; border-radius: var(--radius-lg); padding: 32px; box-shadow: var(--shadow); margin-bottom: 24px; }
        .upload-card h2 { margin-bottom: 24px; color: var(--primary); }
        .upload-form { display: flex; flex-direction: column; gap: 20px; }
        .form-row { display: flex; gap: 16px; align-items: end; }
        .form-group { flex: 1; }
        .form-group label { display: block; margin-bottom: 8px; font-weight: 500; }
        .form-group input, .form-group select { width: 100%; padding: 12px; border: 2px solid var(--gray-200); border-radius: var(--radius); }
        .form-group input[type="file"] { padding: 10px; }
        .message { padding: 16px; border-radius: var(--radius); margin-bottom: 20px; }
        .message.success { background: #dcfce7; color: #166534; border: 1px solid #bbf7d0; }
        .message.error { background: #fef2f2; color: #dc2626; border: 1px solid #fecaca; }
        .files-grid { display: grid; grid-template-columns: repeat(auto-fill, minmax(150px, 1fr)); gap: 16px; }
        .file-item { background: var(--gray-50); border-radius: var(--radius); padding: 12px; text-align: center; }
        .file-item img { max-width: 100%; height: 80px; object-fit: contain; margin-bottom: 8px; }
        .file-item p { font-size: 12px; word-break: break-all; color: var(--gray-600); }
        .logo-info { background: #fef3c7; border: 1px solid #fcd34d; padding: 16px; border-radius: var(--radius); margin-bottom: 20px; }
        .logo-info h4 { color: #92400e; margin-bottom: 8px; }
        .logo-info ul { margin: 0; padding-left: 20px; color: #92400e; font-size: 14px; }
    </style>
</head>
<body>
    <div class="upload-page">
        <div class="upload-container">
            <a href="../" style="color: var(--primary); text-decoration: none; display: inline-flex; align-items: center; gap: 8px; margin-bottom: 24px;">
                &larr; Back to Multicure
            </a>
            
            <h1 style="margin-bottom: 32px;">Multicure Network - File Uploads</h1>
            
            <?php if ($message): ?>
            <div class="message <?php echo $messageType; ?>"><?php echo htmlspecialchars($message); ?></div>
            <?php endif; ?>
            
            <div class="upload-card">
                <h2>Upload Files</h2>
                
                <div class="logo-info">
                    <h4>Logo Naming Convention</h4>
                    <ul>
                        <li>omnicure-logo.png - Main OmniCure logo</li>
                        <li>tyrecure-logo.png - Tyrecure logo</li>
                        <li>dunlopboots-logo.png - Dunlop Boots logo</li>
                        <li>germcure-logo.png - Germcure logo</li>
                        <li>farmcure-logo.png - Farmcure logo</li>
                    </ul>
                </div>
                
                <form class="upload-form" method="POST" enctype="multipart/form-data">
                    <div class="form-row">
                        <div class="form-group">
                            <label>Upload Type</label>
                            <select name="upload_type">
                                <option value="logo">Logo (for sites)</option>
                                <option value="general">General File</option>
                            </select>
                        </div>
                        <div class="form-group">
                            <label>Select File</label>
                            <input type="file" name="file" required accept="image/*,.pdf">
                        </div>
                        <button type="submit" class="btn btn-primary">Upload</button>
                    </div>
                </form>
            </div>
            
            <div class="upload-card">
                <h2>Uploaded Logos</h2>
                <div class="files-grid">
                    <?php if (empty($logos)): ?>
                        <p style="color: var(--gray-600);">No logos uploaded yet.</p>
                    <?php else: ?>
                        <?php foreach ($logos as $logo): ?>
                        <div class="file-item">
                            <img src="logos/<?php echo basename($logo); ?>" alt="">
                            <p><?php echo basename($logo); ?></p>
                        </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>
            
            <div class="upload-card">
                <h2>General Uploads</h2>
                <div class="files-grid">
                    <?php 
                    $generalFiles = array_filter($files, function($f) use ($logosDir) {
                        return strpos($f, '/logos/') === false;
                    });
                    if (empty($generalFiles)): ?>
                        <p style="color: var(--gray-600);">No files uploaded yet.</p>
                    <?php else: ?>
                        <?php foreach ($generalFiles as $file): ?>
                        <div class="file-item">
                            <?php if (preg_match('/\.(jpg|jpeg|png|gif|webp)$/i', $file)): ?>
                                <img src="<?php echo basename($file); ?>" alt="">
                            <?php else: ?>
                                <div style="height: 80px; display: flex; align-items: center; justify-content: center; color: var(--gray-400);">PDF</div>
                            <?php endif; ?>
                            <p><?php echo basename($file); ?></p>
                        </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</body>
</html>
